<?php

namespace App\LedgerReports;

use Abivia\Ledger\Messages\Report;
use Abivia\Ledger\Models\ReportData;
use Abivia\Ledger\Reports\AbstractReport;

use Abivia\Ledger\Models\LedgerDomain;
use Abivia\Ledger\Models\LedgerCurrency;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\JournalDetail;
use App\Models\CustomLedgerAccount;
use App\Models\JournalDetailOverwrite;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AccountStatementReport extends AbstractReport
{
    protected Report $reportMessage;

    public function collect(Report $report): ReportData
    {
        $report->validate();
        $this->reportMessage = $report;

        $ledgerDomain = LedgerDomain::findWith($report->domain)->firstOrFail();
        $currency = LedgerCurrency::findOrBreaker($report->currency);

        $reportData = new AccountStatementData();
        $reportData->request = $report;
        $reportData->decimals = $currency->decimals;

        $account_code = $report->options['account'] ?? null;
        if ($account_code) {
            $account = CustomLedgerAccount::where("code", $account_code)->first();
            $reportData->account = $account;

            $from = $report->fromDate ?? '1900-01-01';
            $to = $report->toDate ?? now()->toDateString();




            // Get transactions within the date range
            $details = DB::table('journal_details')
                ->select(
                    'journal_details.journalDetailId',
                    'journal_details.amount',
                    'entries.transDate',
                    'entries.description',
                    'entries.journalReferenceUuid',

                    'entries.reviewed',
                    'entries.extra',
                    'accounts.code as account_code',
                    'accounts.ledgerUuid as account_uuid',
                    'transfers.id as transfer_id',
                    'transfers.type as transfer_type',
                    'transfers.reference as transfer_number',
                    'transfers.send_fee',
                    'transfers.receiver_fee',
                    'users.name as user_name',
                    'senders.name as sender_name',
                    'receivers.name as receiver_name'
                )
                ->join('journal_entries as entries', 'journal_details.journalEntryId', '=', 'entries.journalEntryId')
                ->join('ledger_accounts as accounts', 'journal_details.ledgerUuid', '=', 'accounts.ledgerUuid')
                ->leftJoin('ledger_transfers as transfers', 'entries.extra->exchange_id', '=', 'transfers.id')
                ->leftJoin('users', 'transfers.user_id', '=', 'users.id')
                ->leftJoin('clients as senders', 'transfers.sender_id', '=', 'senders.id')
                ->leftJoin('clients as receivers', 'transfers.receiver_id', '=', 'receivers.id')
                ->whereBetween('entries.transDate', [$from, $to])
                ->orderBy('entries.transDate')
                ->get();

            $reportData->lines = $details->sortBy(function ($detail) {
                return $detail->transDate;
            })->values();
            $reportData->openingBalance =  $account->getBalanceByDate(Carbon::parse($from)->toDateString(), $this->reportMessage->currency);


            return $reportData;
        } else {
            Log::error('Account code not provided in report options');
            throw new \Exception('Account code is required for account statement report.');
        }
    }

    public function prepare(ReportData $reportData): Collection
    {
        /** @var AccountStatementData $data */
        $data = $reportData;

        $balance = $data->openingBalance * (-1);
        $decimals = $data->decimals;
        $totalDebit = '0';
        $totalCredit = '0';

        $rows = collect();

        // Sort lines by transDate
        $sortedLines = $data->lines->sortBy(function ($detail) {
            return $detail->transDate;
        });

        $counter = 1;
        $in_transfers = 0;
        $out_transfers = 0;

        foreach ($sortedLines as $detail) {
            // Use direct properties from the DB query
            $debit = $detail->amount < 0 ? bcmul($detail->amount, '-1', $decimals) : '0';
            $credit = $detail->amount >= 0 ? $detail->amount : '0';

            $totalDebit = bcadd($totalDebit, $debit, $decimals);
            $totalCredit = bcadd($totalCredit, $credit, $decimals);

            $amountChange = bcsub($debit, $credit, $decimals);
            $balance = bcadd($balance, $amountChange, $decimals);

            $debit !== '0' ? $out_transfers++ : $in_transfers++;

            $rows->push([
                '#' => $counter++,
                'id' => $detail->journalDetailId,
                'date' => \Carbon\Carbon::parse($detail->transDate),
                'transfer_id' => $detail->id ?? "",
                'user' => $detail->user_name ?? 'اليوزر',
                'transfer_side' => !empty($detail->user_name) ? __('agent') : __("administrator"),
                'account' => $detail->account_code ?? $detail->account_uuid,
                'transfer_number' => $detail->transfer_number ?? "",
                'sender' => $detail->sender_name ?? "",
                'receiver' => $detail->receiver_name ?? "",
                'send_fee' => $detail->send_fee ?? "",
                'delivery_fee' => $detail->receiver_fee ?? "",
                'type' => $detail->transfer_type ?? "",
                'document' => $detail->journalReferenceUuid ?? "",
                'debit' => $debit !== '0' ? $debit : '',
                'credit' => $credit !== '0' ? $credit : '',
                'client' => $detail->extra ?? '-',
                'description' => $detail->description ?? '',
                'amount' => $debit !== '0' ? $debit : $credit,
                'balance' => $balance,
                'currency' => $this->reportMessage->currency,
                'review' => $detail->reviewed ?? 0,
                'transfer_type' => ($detail->transfer_type ?? "") . " " . ($debit !== '0' ? __('out') : __('in')),
                'direction' => $debit !== '0' ? __('out') : __('in')
            ]);
        }

        // Totals row (optional)
        $totalsRow = [
            '#' => 0,
            'id' => 0,
            'date' => '',
            'user' => '',
            'transfer_side' => '',
            'account' => "",
            'type' => '',
            'document' => '',
            'debit' => $totalDebit,
            'credit' => $totalCredit,
            'client' => '',
            'description' => __('إجمالي الحركة'),
            'amount' => '',
            'balance' => '',
            'currency' => $this->reportMessage->currency,
            'review' => 1,
        ];

        // $rows->push($totalsRow);

        // Update report data summary
        $data->totalDebit = $totalDebit;
        $data->totalCredit = $totalCredit;
        $data->entryCount = $rows->count();
        $data->outTransfers = $out_transfers;
        $data->inTransfers = $in_transfers;
        $data->currentBalance = $balance;
        $data->last_id = $counter++;

        Log::info("Report summary: totalDebit={$totalDebit}, totalCredit={$totalCredit}, entries={$data->entryCount}");

        return $rows;
    }
}
